<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

use App\Http\Controllers\ProfileController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\RoleController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\PlanController;
use App\Http\Controllers\CarterasController;
use App\Http\Controllers\ReportesController;
use App\Http\Controllers\ModuleController;
use App\Http\Controllers\VodafoneController;
use App\Http\Controllers\VodafoneCrudController;
use App\Http\Controllers\VodafoneImportController;
use App\Http\Controllers\EnriquecimientoController;
use App\Http\Controllers\PublicProfileController;
use Inertia\Inertia;

// ===================
// RUTA INICIAL
// ===================
Route::get('/', function () {
    return redirect()->route(Auth::check() ? 'dashboard' : 'login');
});

// ===================
// AUTENTICACIÓN
// ===================
require __DIR__ . '/auth.php';

// ===================
// RUTAS PROTEGIDAS
// ===================
Route::middleware(['auth', 'session.control'])->group(function () {

    // ===================
    // PERFIL
    // ===================
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');
    Route::post('/profile/avatar', [ProfileController::class, 'updateAvatar'])->name('profile.avatar.update');
    Route::put('/profile/password', [ProfileController::class, 'updatePassword'])->name('profile.password.update');

    // ===================
    // GESTIÓN DE SESIONES - SIMPLE
    // ===================
    Route::get('/profile/sessions', [App\Http\Controllers\SessionController::class, 'index'])->name('profile.sessions');
    Route::get('/sessions/status', [App\Http\Controllers\SessionController::class, 'getSessionStatus'])->name('session.status');
    Route::post('/sessions/close-others', [App\Http\Controllers\SessionController::class, 'closeOtherSessions'])->name('session.close-others');
    Route::post('/sessions/confirm-limit', [App\Http\Controllers\SessionController::class, 'confirmLimit'])->name('session.confirm-limit');
    Route::post('/sessions/logout', [App\Http\Controllers\SessionController::class, 'logout'])->name('session.logout');

    // ===================
    // DASHBOARD
    // ===================
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/dashboard/reporte/{id}', [DashboardController::class, 'showReporte'])
        ->name('dashboard.reporte')
        ->where('id', '[0-9]+')
        ->middleware(['can:reportes.ver', \App\Http\Middleware\CheckReporteAccess::class]);

    // ===================
    // ROLES
    // ===================
    Route::middleware('can:roles.ver')->get('/roles', [RoleController::class, 'index'])->name('roles.index');
    Route::middleware('can:roles.crear')->post('/roles', [RoleController::class, 'store'])->name('roles.store');
    Route::middleware('can:roles.editar')->put('/roles/{role}', [RoleController::class, 'update'])->name('roles.update');
    Route::middleware('can:roles.eliminar')->delete('/roles/{role}', [RoleController::class, 'destroy'])->name('roles.destroy');
    // MÓDULOS (desde pestaña Roles)
    Route::middleware('can:roles.crear')->post('/modulos', [ModuleController::class, 'store'])->name('modules.store');

    // ===================
    // USUARIOS
    // ===================
    Route::middleware('can:usuarios.ver')->get('/users', [UserController::class, 'index'])->name('users.index');
    Route::middleware('can:usuarios.crear')->post('/users', [UserController::class, 'store'])->name('users.store');
    Route::middleware('can:usuarios.editar')->put('/users/{user}', [UserController::class, 'update'])->name('users.update');
    Route::middleware('can:usuarios.eliminar')->delete('/users/{user}', [UserController::class, 'destroy'])->name('users.destroy');

    // ===================
    // PLANES
    // ===================
    Route::middleware('can:planes.ver')->get('/planes', [PlanController::class, 'index'])->name('planes.index');
    Route::middleware('can:planes.crear')->post('/planes', [PlanController::class, 'store'])->name('planes.store');
    Route::middleware('can:planes.editar')->put('/planes/{plan}', [PlanController::class, 'update'])->name('planes.update');
    Route::middleware('can:planes.eliminar')->delete('/planes/{plan}', [PlanController::class, 'destroy'])->name('planes.destroy');

    // ===================
    // CARTERAS
    // ===================
    Route::middleware('can:carteras.ver')->get('/carteras', [CarterasController::class, 'index'])->name('carteras.index');
    Route::middleware('can:carteras.crear')->post('/carteras', [CarterasController::class, 'store'])->name('carteras.store');
    Route::middleware('can:carteras.editar')->put('/carteras/{cartera}', [CarterasController::class, 'update'])->name('carteras.update');
    Route::middleware('can:carteras.eliminar')->delete('/carteras/{cartera}', [CarterasController::class, 'destroy'])->name('carteras.destroy');

    // ===================
    // REPORTES
    // ===================
    Route::middleware('can:reportes.ver')->get('/reportes', [ReportesController::class, 'index'])->name('reportes.index');
    Route::middleware('can:reportes.crear')->post('/reportes', [ReportesController::class, 'store'])->name('reportes.store');
    Route::middleware('can:reportes.editar')->put('/reportes/{reporte}', [ReportesController::class, 'update'])->name('reportes.update');
    Route::middleware('can:reportes.eliminar')->delete('/reportes/{reporte}', [ReportesController::class, 'destroy'])->name('reportes.destroy');

    // ===================
    // VODAFONE
    // ===================
    Route::middleware('can:vodafone.ver')->match(['GET', 'POST'], '/vodafone', [VodafoneController::class, 'index'])->name('vodafone.index');
    Route::middleware('can:vodafone.crear')->post('/vodafone/crear', [VodafoneController::class, 'store'])->name('vodafone.store');
    Route::middleware('can:vodafone.editar')->put('/vodafone/{vodafone}', [VodafoneController::class, 'update'])->name('vodafone.update');
    Route::middleware('can:vodafone.eliminar')->delete('/vodafone/{vodafone}', [VodafoneController::class, 'destroy'])->name('vodafone.destroy');
    Route::middleware('can:vodafone.asignar')->post('/vodafone/asignar', [VodafoneController::class, 'asignar'])->name('vodafone.asignar');
    Route::middleware('can:vodafone.agendar')->post('/vodafone/{vodafone}/agendar', [VodafoneController::class, 'agendar'])->name('vodafone.agendar');
    Route::middleware('can:vodafone.importar')->get('/vodafone/plantilla-excel', [VodafoneController::class, 'descargarPlantilla'])->name('vodafone.plantilla');
    Route::middleware('can:vodafone.importar')->post('/vodafone/preview', [VodafoneImportController::class, 'preview'])->name('vodafone.preview');
    Route::middleware('can:vodafone.importar')->post('/vodafone/importar-confirmado', [VodafoneImportController::class, 'importarConfirmado'])->name('vodafone.importarConfirmado');
    Route::middleware('can:vodafone.importar')->get('/vodafone/errores-log/{id}', [VodafoneImportController::class, 'obtenerErroresLog'])->name('vodafone.obtenerErroresLog');

    // ===================
    // VODAFONE CRUD SIMPLE
    // ===================
    Route::middleware('can:vodafone.ver')->match(['GET', 'POST'], '/vodafone-crud', [VodafoneCrudController::class, 'index'])->name('vodafone-crud.index');
    Route::middleware('can:vodafone.crear')->post('/vodafone-crud/crear', [VodafoneCrudController::class, 'store'])->name('vodafone-crud.store');
    Route::middleware('can:vodafone.editar')->put('/vodafone-crud/{vodafone}', [VodafoneCrudController::class, 'update'])->name('vodafone-crud.update');
    Route::middleware('can:vodafone.eliminar')->delete('/vodafone-crud/{vodafone}', [VodafoneCrudController::class, 'destroy'])->name('vodafone-crud.destroy');

    // ===================
    // ENRIQUECIMIENTO OPERADOR
    // ===================
    Route::middleware('can:enriquecimiento.ver')->get('/enriquecimiento', [EnriquecimientoController::class, 'index'])->name('enriquecimiento.index');
    Route::middleware('can:enriquecimiento.ver')->post('/enriquecimiento/filtrar', [EnriquecimientoController::class, 'filtrar'])->name('enriquecimiento.filtrar');
    Route::middleware('can:enriquecimiento.crear')->post('/enriquecimiento/importar-manual', [EnriquecimientoController::class, 'importarManual'])->name('enriquecimiento.importar.manual');
    Route::middleware('can:enriquecimiento.ver')->get('/enriquecimiento/{importacion}/detalles', [EnriquecimientoController::class, 'detalles'])->name('enriquecimiento.detalles');
    Route::middleware('can:enriquecimiento.eliminar')->delete('/enriquecimiento/{importacion}', [EnriquecimientoController::class, 'destroy'])->name('enriquecimiento.destroy');
    Route::middleware('can:enriquecimiento.pausar')->post('/enriquecimiento/{importacion}/pausar', [EnriquecimientoController::class, 'pausar'])->name('enriquecimiento.pausar');
    Route::middleware('can:enriquecimiento.importar')->post('/enriquecimiento/importar-excel', [EnriquecimientoController::class, 'importarExcel'])->name('enriquecimiento.importar.excel');
    Route::middleware('can:enriquecimiento.importar')->get('/enriquecimiento/plantilla-excel', [EnriquecimientoController::class, 'descargarPlantilla'])->name('enriquecimiento.plantilla');
    Route::middleware('can:enriquecimiento.exportar')->get('/enriquecimiento/{importacion}/exportar', [EnriquecimientoController::class, 'exportar'])->name('enriquecimiento.exportar');
    Route::middleware('can:enriquecimiento.ver')->get('/enriquecimiento/datos-actualizados', [EnriquecimientoController::class, 'obtenerDatosActualizados'])->name('enriquecimiento.datos.actualizados');
    Route::middleware('can:enriquecimiento.ver')->get('/enriquecimiento/{importacion}/detalles-actualizados', [EnriquecimientoController::class, 'obtenerDetallesActualizados'])->name('enriquecimiento.detalles.actualizados');

    // Rutas de reprocesamiento
    Route::middleware('can:enriquecimiento.crear')->post('/enriquecimiento/{importacion}/reprocesar-rechazados', [EnriquecimientoController::class, 'reprocesarRechazados'])->name('enriquecimiento.reprocesar.rechazados');
    Route::middleware('can:enriquecimiento.crear')->post('/enriquecimiento/{numero}/reprocesar', [EnriquecimientoController::class, 'reprocesarNumero'])->name('enriquecimiento.reprocesar');
    Route::middleware('can:enriquecimiento.crear')->post('/enriquecimiento/{importacion}/reprocesar-todos', [EnriquecimientoController::class, 'reprocesarTodos'])->name('enriquecimiento.reprocesar.todos');

    // Rutas para el bot (sin middleware de permisos)
    Route::post('/api/enriquecimiento/procesar', [EnriquecimientoController::class, 'procesarBot'])->name('enriquecimiento.procesar.bot');
    Route::post('/api/enriquecimiento/actualizar-estado', [EnriquecimientoController::class, 'actualizarEstadoBot'])->name('enriquecimiento.actualizar.estado');

    // ===================
    // PERFILES PÚBLICOS
    // ===================

    Route::middleware('can:profiles.ver-directorio')->get('/profiles', [PublicProfileController::class, 'index'])->name('profiles.index');
    Route::middleware('can:profiles.ver-perfil-publico')->get('/profiles/{user}', [PublicProfileController::class, 'show'])->name('profiles.show');

    // ===================
    // DEBUG DE SESIONES
    // ===================
    Route::get('/debug-sessions', function () {
        /** @var \App\Models\User $user */
        $user = Auth::user();
        $sesionesActivas = $user->getSesionesActivasCount();
        $limiteSesiones = $user->getLimiteSesiones();

        return response()->json([
            'user_id' => $user->id,
            'user_name' => $user->name,
            'active_sessions_count' => $sesionesActivas,
            'sessions_limit' => $limiteSesiones,
            'limit_exceeded' => $sesionesActivas > $limiteSesiones,
            'plan' => $user->plan ? [
                'nombre' => $user->plan->nombre,
                'max_sesiones' => $user->plan->max_sesiones
            ] : [
                'nombre' => 'Sin Plan',
                'max_sesiones' => $limiteSesiones
            ],
            'user_sessions' => $user->userSessions()->get()->map(function ($session) {
                return [
                    'id' => $session->id,
                    'session_id' => $session->session_id,
                    'device_name' => $session->device_name,
                    'ip_address' => $session->ip_address,
                    'last_activity' => $session->last_activity,
                    'is_active' => $session->isActive()
                ];
            })
        ]);
    })->name('debug.sessions');
});

// ===================
// RUTA BACKUP SI FALLA
// ===================
Route::fallback(function () {
    return redirect()->route(Auth::check() ? 'dashboard' : 'login');
});
