<?php

/**
 * Cliente PHP para Bot de Genius Business
 * 
 * Este archivo muestra cómo conectarse a la API del bot
 * desde un sistema externo usando PHP.
 */

class GeniusBusinessBotClient
{
    private $apiUrl;
    private $timeout;

    public function __construct($apiUrl, $timeout = 30)
    {
        $this->apiUrl = rtrim($apiUrl, '/');
        $this->timeout = $timeout;
    }

    /**
     * Procesar números telefónicos
     */
    public function procesarNumeros($userId, $numeros, $metadata = [])
    {
        $endpoint = '/api/enriquecimiento/procesar';

        $data = [
            'user_id' => $userId,
            'numeros' => $numeros,
            'metadata' => $metadata
        ];

        return $this->makeRequest('POST', $endpoint, $data);
    }

    /**
     * Actualizar estado de un número
     */
    public function actualizarEstado($numeroId, $estado, $datosEnriquecidos = [], $errorMensaje = null)
    {
        $endpoint = '/api/enriquecimiento/actualizar-estado';

        $data = [
            'numero_id' => $numeroId,
            'estado' => $estado,
            'datos_enriquecidos' => $datosEnriquecidos
        ];

        if ($errorMensaje) {
            $data['error_mensaje'] = $errorMensaje;
        }

        return $this->makeRequest('POST', $endpoint, $data);
    }

    /**
     * Realizar petición HTTP
     */
    private function makeRequest($method, $endpoint, $data = [])
    {
        $url = $this->apiUrl . $endpoint;

        $headers = [
            'Content-Type: application/json',
            'Accept: application/json'
        ];

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_CUSTOMREQUEST => $method
        ]);

        if (!empty($data)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new Exception("cURL Error: " . $error);
        }

        $decodedResponse = json_decode($response, true);

        return [
            'http_code' => $httpCode,
            'success' => $httpCode >= 200 && $httpCode < 300,
            'data' => $decodedResponse,
            'raw_response' => $response
        ];
    }
}

// ======================
// EJEMPLO DE USO
// ======================

try {
    // Configuración
    $client = new GeniusBusinessBotClient('http://localhost:8000');

    // Ejemplo 1: Procesar números
    echo "=== Procesando números ===\n";
    $response = $client->procesarNumeros(
        1, // ID del usuario
        ['612345678', '723456789', '634567890'],
        ['fuente' => 'sistema_externo', 'prioridad' => 'alta']
    );

    if ($response['success']) {
        echo "✓ Números procesados exitosamente\n";
        echo "ID de Importación: " . $response['data']['importacion_id'] . "\n";
        echo "Créditos utilizados: " . $response['data']['creditos_utilizados'] . "\n";
        echo "Créditos restantes: " . $response['data']['creditos_restantes'] . "\n";
    } else {
        echo "✗ Error al procesar números: " . $response['data']['error'] . "\n";
    }

    // Simular espera para procesamiento
    sleep(2);

    // Ejemplo 2: Actualizar estado
    echo "\n=== Actualizando estado ===\n";
    $numeroId = 123; // ID del número a actualizar

    $response = $client->actualizarEstado(
        $numeroId,
        'procesado',
        [
            'operador' => 'Movistar',
            'region' => 'Madrid',
            'tipo' => 'prepago',
            'activo' => true
        ]
    );

    if ($response['success']) {
        echo "✓ Estado actualizado exitosamente\n";
        echo "Número: " . $response['data']['numero_actualizado'] . "\n";
        echo "Nuevo estado: " . $response['data']['estado'] . "\n";
    } else {
        echo "✗ Error al actualizar estado: " . $response['data']['error'] . "\n";
    }
} catch (Exception $e) {
    echo "Error fatal: " . $e->getMessage() . "\n";
}
