# Ejemplo de Integración con Bot

Este directorio contiene ejemplos de código para integrar sistemas externos con la API del bot de Genius Business.

## 🚀 ¿Cómo usar el Bot API?

### 📍 **Paso 1: Configurar tu Servidor**

**Si usas LARAGON (recomendado):**
```bash
# Tu URL será automáticamente:
http://genius-business.test/api/enriquecimiento/procesar-bot
```

**Si usas `php artisan serve`:**
```bash
# Ejecuta en la raíz del proyecto:
php artisan serve --host=0.0.0.0 --port=8000

# Tu URL será:
http://localhost:8000/api/enriquecimiento/procesar-bot
```

**Si usas un servidor web (Apache/Nginx):**
```bash
# Tu URL dependerá de tu configuración:
http://tu-dominio.com/api/enriquecimiento/procesar-bot
```

### 📍 **Paso 2: Configurar Variables de Entorno**

En tu archivo `.env` configura:
```env
# Configuración del Bot (SIN token de autenticación)
BOT_ENABLED=true
BOT_MAX_CONCURRENT_JOBS=5
BOT_RETRY_ATTEMPTS=3
BOT_TIMEOUT_SECONDS=30
BOT_URL=http://localhost:8000  # ⚠️ CAMBIA por tu URL real
```

### 📍 **Paso 3: Endpoints Disponibles**

**🔹 Procesar números desde bot externo:**
```http
POST /api/enriquecimiento/procesar-bot
Content-Type: application/json

{
    "user_id": 1,
    "numeros": ["+57123456789", "+57987654321"],
    "metadata": {
        "origen": "mi_bot",
        "prioridad": "alta"
    }
}
```

**🔹 Actualizar estado de número procesado:**
```http
POST /api/enriquecimiento/actualizar-estado
Content-Type: application/json

{
    "numero_id": 123,
    "estado": "procesado",
    "datos_enriquecidos": {
        "operador": "Claro",
        "tipo": "movil",
        "region": "Bogotá",
        "activo": true
    }
}
```

### 📍 **Paso 4: Ejemplos Prácticos**

**🔸 Desde tu aplicación PHP:**
```php
$url = 'http://localhost:8000/api/enriquecimiento/procesar-bot';
$data = [
    'user_id' => 1,
    'numeros' => ['+57123456789', '+57987654321'],
    'metadata' => ['origen' => 'mi_sistema']
];

$ch = curl_init($url);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

$response = curl_exec($ch);
curl_close($ch);

echo $response;
```

**🔸 Desde la línea de comandos:**
```bash
# Reemplaza localhost:8000 por tu URL real
curl -X POST http://localhost:8000/api/enriquecimiento/procesar-bot \
  -H "Content-Type: application/json" \
  -d '{"user_id": 1, "numeros": ["+57123456789"]}'
```

**🔸 Desde JavaScript/Node.js:**
```javascript
const response = await fetch('http://localhost:8000/api/enriquecimiento/procesar-bot', {
    method: 'POST',
    headers: {
        'Content-Type': 'application/json',
    },
    body: JSON.stringify({
        user_id: 1,
        numeros: ['+57123456789', '+57987654321'],
        metadata: { origen: 'mi_bot' }
    })
});

const data = await response.json();
console.log(data);
```

## 📂 Archivos de Ejemplo

- `bot_client_example.php` - Cliente PHP completo con ejemplos
- `test_bot_api.sh` - Script Bash para probar los endpoints

## 🔧 Configuración según tu Entorno

### 🟢 **LARAGON (Windows)**
```env
BOT_URL=http://genius-business.test
```
URL de prueba: `http://genius-business.test/api/enriquecimiento/procesar-bot`

### 🟢 **XAMPP/WAMP (Windows)**
```env
BOT_URL=http://localhost/genius-business/public
```
URL de prueba: `http://localhost/genius-business/public/api/enriquecimiento/procesar-bot`

### 🟢 **php artisan serve**
```env
BOT_URL=http://localhost:8000
```
URL de prueba: `http://localhost:8000/api/enriquecimiento/procesar-bot`

### 🟢 **Servidor de Producción**
```env
BOT_URL=https://tu-dominio.com
```
URL de prueba: `https://tu-dominio.com/api/enriquecimiento/procesar-bot`

## 🛠️ Pruebas y Depuración

**Ver logs en tiempo real:**
```bash
tail -f storage/logs/laravel.log
```

**Probar conectividad:**
```bash
# Ejecutar script de pruebas
cd examples
chmod +x test_bot_api.sh
./test_bot_api.sh
```

**Verificar que el servidor esté corriendo:**
```bash
curl -I http://localhost:8000/api/enriquecimiento/procesar-bot
```

## ⚠️ **Importante**

- ✅ **NO requiere autenticación** - Puedes usar la API directamente
- ✅ **Cambia la URL** según tu entorno (localhost, dominio, etc.)
- ✅ **Verifica que Laravel esté corriendo** antes de hacer peticiones
- ✅ **Los números deben tener formato válido** (ej: +57123456789)
