<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;
use App\Models\Cartera;
use App\Models\Reporte;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasRoles, SoftDeletes;

    /**
     * Scope para filtrar solo usuarios activos
     */
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }



    protected $fillable = [
        'name',
        'email',
        'password',
        'active',
        'tipo_documento_id',
        'plan_id',
        'numero_documento',
        'is_public_profile',
        'bio',
        'avatar',
        'phone',
        'position',
        'department',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password'          => 'hashed',
        'active'            => 'boolean',
        'is_public_profile' => 'boolean',
    ];

    /**
     * Relación pivote user ⇆ reportes
     */
    public function reportes()
    {
        return $this
            ->belongsToMany(Reporte::class, 'reporte_user')
            ->with('cartera');
    }

    /**
     * Relación pivote user ⇆ carteras
     */
    public function carteras()
    {
        return $this
            ->belongsToMany(Cartera::class, 'cartera_user');
    }
    public function getEffectiveCarteras()
    {
        return $this->roles->flatMap->carteras
            ->merge($this->carteras)
            ->unique('id')
            ->filter(fn($cartera) => $cartera->estado)
            ->values();
    }


    public function getEffectiveReportes()
    {
        return $this->roles->flatMap->reportes
            ->merge($this->reportes)
            ->unique('id')
            ->values();
    }

    public function importacionesVodafone()
    {
        return $this->hasMany(LogImportacionVodafone::class);
    }

    // Relación con TipoDocumento
    public function tipoDocumento()
    {
        return $this->belongsTo(TipoDocumento::class, 'tipo_documento_id');
    }

    // Relación con Plan
    public function plan()
    {
        return $this->belongsTo(Plan::class);
    }

    // Relación con sesiones de usuario
    public function userSessions()
    {
        return $this->hasMany(UserSession::class);
    }

    // Sesiones activas del usuario
    public function sesionesActivas()
    {
        return $this->userSessions()->activas();
    }

    // Accessor para obtener el documento completo
    public function getDocumentoCompletoAttribute()
    {
        if (!$this->tipoDocumento || !$this->numero_documento) {
            return null;
        }
        return $this->tipoDocumento->codigo . ': ' . $this->numero_documento;
    }

    // Accessor para obtener la URL completa del avatar
    public function getAvatarUrlAttribute()
    {
        if (!$this->avatar) {
            return null;
        }
        return asset('storage/' . $this->avatar);
    }

    /**
     * Verifica si el usuario ha alcanzado el límite de sesiones
     */
    public function haAlcanzadoLimiteSesiones()
    {
        if (!$this->plan) {
            return false; // Sin plan = sin límite
        }

        $sesionesActivas = $this->sesionesActivas()->count();
        return $sesionesActivas > $this->plan->max_sesiones;
    }

    /**
     * Obtiene el número de sesiones activas
     */
    public function getSesionesActivasCount()
    {
        return $this->sesionesActivas()->count();
    }

    /**
     * Obtiene el límite de sesiones según su plan
     */
    public function getLimiteSesiones()
    {
        return $this->plan ? $this->plan->max_sesiones : 1; // 1 sesión por defecto si no tiene plan
    }

    /**
     * Cierra todas las sesiones excepto la actual
     */
    /**
     * Cierra todas las sesiones excepto la actual
     */
    public function cerrarOtrasSesiones($currentSessionId = null)
    {
        $query = $this->userSessions();

        if ($currentSessionId) {
            $query->where('session_id', '!=', $currentSessionId);
        }

        // Obtener las sesiones que vamos a cerrar antes de borrarlas
        $sesionesACerrar = $query->get();

        // Invalidar las sesiones de Laravel
        foreach ($sesionesACerrar as $sesion) {
            if ($sesion->session_id) {
                // Eliminar de la tabla sessions de Laravel
                DB::table('sessions')->where('id', $sesion->session_id)->delete();
            }
        }

        // Borrar de la tabla user_sessions
        return $this->userSessions()
            ->when($currentSessionId, function ($q) use ($currentSessionId) {
                $q->where('session_id', '!=', $currentSessionId);
            })
            ->delete();
    }
}
