<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class EnriquecimientoNumero extends Model
{
    use HasFactory;

    protected $table = 'enriquecimiento_numeros';

    protected $fillable = [
        'importacion_id',
        'numero_telefono',
        'numero_normalizado',
        'operador_origen',
        'operador_actual',
        'fecha_actualizacion_operador',
        'estado',
        'datos_adicionales',
        'error_detalle',
        'fecha_procesamiento',
        'fecha_registro'
    ];

    protected $casts = [
        'datos_adicionales' => 'array',
        'fecha_actualizacion_operador' => 'datetime',
        'fecha_procesamiento' => 'datetime',
        'fecha_registro' => 'datetime'
    ];

    protected $dates = [
        'fecha_actualizacion_operador',
        'fecha_procesamiento',
        'fecha_registro'
    ];

    /**
     * Relación con la importación
     */
    public function importacion(): BelongsTo
    {
        return $this->belongsTo(EnriquecimientoImportacion::class, 'importacion_id');
    }

    /**
     * Scope para filtrar por estado
     */
    public function scopeByEstado($query, $estado)
    {
        return $query->where('estado', $estado);
    }

    /**
     * Scope para filtrar números pendientes
     */
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    /**
     * Scope para filtrar números procesados
     */
    public function scopeProcesados($query)
    {
        return $query->where('estado', 'completado');
    }

    /**
     * Normalizar número de teléfono
     */
    public function normalizarNumero()
    {
        $numero = preg_replace('/[^0-9+]/', '', $this->numero_telefono);

        // Lógica de normalización según país/formato
        if (strlen($numero) == 10 && !str_starts_with($numero, '+')) {
            $numero = '+51' . $numero; // Ejemplo para Perú
        }

        $this->numero_normalizado = $numero;
        return $numero;
    }
}
