<?php

namespace App\Jobs;

use App\Models\EnriquecimientoImportacion;
use App\Models\EnriquecimientoNumero;
use App\Models\UsuarioCredito;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class ProcesarEnriquecimientoNumerosJob implements ShouldQueue
{
    use Queueable;

    private EnriquecimientoImportacion $importacion;

    public function __construct(EnriquecimientoImportacion $importacion)
    {
        $this->importacion = $importacion;
    }

    public function handle(): void
    {
        try {
            // Verificar que la importación existe
            if (!$this->importacion || !$this->importacion->exists) {
                Log::error("Importación no encontrada o eliminada");
                return;
            }

            // Verificar que la importación no esté cancelada
            $this->importacion->refresh();
            if (in_array($this->importacion->estado, ['cancelado', 'rechazado', 'procesado'])) {
                Log::info("Job terminado - Importación ya está en estado final", [
                    'importacion_id' => $this->importacion->id,
                    'estado' => $this->importacion->estado
                ]);
                return;
            }

            Log::info("Procesando números", [
                'importacion_id' => $this->importacion->id
            ]);

            // Obtener números pendientes
            $numerosAProcesar = EnriquecimientoNumero::where('importacion_id', $this->importacion->id)
                ->where('estado', 'pendiente')
                ->take(50)
                ->get();

            if ($numerosAProcesar->isEmpty()) {
                // Marcar como completado si no hay más pendientes
                $this->importacion->update(['estado' => 'procesado', 'fecha_completado' => now()]);
                return;
            }

            $numerosProcesados = 0;

            foreach ($numerosAProcesar as $numero) {
                try {
                    // Verificar si fue cancelado o pausado
                    $this->importacion->refresh();
                    if (in_array($this->importacion->estado, ['rechazado', 'cancelado'])) {
                        Log::info("Procesamiento detenido por estado: {$this->importacion->estado}", [
                            'importacion_id' => $this->importacion->id
                        ]);
                        break;
                    }

                    // Verificar créditos ANTES de procesar cada número
                    if (!UsuarioCredito::puedeUsarCreditos($this->importacion->user_id, 1)) {
                        Log::info("Créditos insuficientes, marcando números restantes como rechazado", [
                            'user_id' => $this->importacion->user_id,
                            'numero_id' => $numero->id
                        ]);

                        // Marcar todos los números pendientes restantes como rechazado por falta de créditos
                        EnriquecimientoNumero::where('importacion_id', $this->importacion->id)
                            ->where('estado', 'pendiente')
                            ->update([
                                'estado' => 'rechazado',
                                'error_detalle' => 'Sin créditos suficientes para procesar',
                                'updated_at' => now()
                            ]);

                        // También cambiar el estado de la importación
                        $this->importacion->update([
                            'estado' => 'procesado', // Cambiar a procesado ya que se completó (aunque sea con rechazos)
                            'fecha_completado' => now(),
                            'observaciones' => 'Procesamiento completado con limitación de créditos'
                        ]);

                        break;
                    }

                    // PROCESAMIENTO SIMPLE - Solo marcar como procesado
                    $numero->update([
                        'estado' => 'procesando',
                        'fecha_procesamiento' => now()
                    ]);

                    // Simular procesamiento básico
                    $numero->update([
                        'estado' => 'procesado',
                        'operador_origen' => 'Detectado',
                        'operador_actual' => 'Detectado',
                        'numero_normalizado' => preg_replace('/[^0-9]/', '', $numero->numero_telefono),
                        'datos_adicionales' => json_encode(['procesado' => true]),
                        'fecha_procesamiento' => now()
                    ]);

                    // Consumir crédito DESPUÉS de procesar exitosamente cada número
                    UsuarioCredito::consumirCreditos(
                        $this->importacion->user_id,
                        1,
                        "Enriquecimiento número {$numero->numero_telefono}"
                    );

                    $numerosProcesados++;

                    Log::info("Número procesado y crédito consumido", [
                        'numero_id' => $numero->id,
                        'numero_telefono' => $numero->numero_telefono,
                        'user_id' => $this->importacion->user_id
                    ]);
                } catch (\Exception $e) {
                    Log::error("Error procesando número", [
                        'numero_id' => $numero->id,
                        'error' => $e->getMessage()
                    ]);

                    $numero->update([
                        'estado' => 'rechazado',
                        'error_detalle' => $e->getMessage()
                    ]);
                }
            }

            // Actualizar estadísticas SOLO si no fue cancelado
            $this->importacion->refresh();
            if (!in_array($this->importacion->estado, ['cancelado', 'rechazado'])) {
                $totalPendientes = EnriquecimientoNumero::where('importacion_id', $this->importacion->id)
                    ->where('estado', 'pendiente')
                    ->count();

                $totalProcesados = EnriquecimientoNumero::where('importacion_id', $this->importacion->id)
                    ->where('estado', 'procesado')
                    ->count();

                $this->importacion->update([
                    'numeros_procesados' => $totalProcesados,
                    'numeros_pendientes' => $totalPendientes,
                    'estado' => $totalPendientes > 0 ? 'procesando' : 'procesado'
                ]);

                if ($totalPendientes === 0) {
                    $this->importacion->update(['fecha_completado' => now()]);
                } else {
                    // Continuar procesando si hay más números
                    ProcesarEnriquecimientoNumerosJob::dispatch($this->importacion)->delay(now()->addSeconds(2));
                }

                Log::info("Lote procesado", [
                    'procesados' => $numerosProcesados,
                    'pendientes' => $totalPendientes
                ]);
            } else {
                Log::info("Procesamiento detenido definitivamente", [
                    'importacion_id' => $this->importacion->id,
                    'estado' => $this->importacion->estado
                ]);
            }
        } catch (\Exception $e) {
            Log::error("Error procesando", ['error' => $e->getMessage()]);

            if ($this->importacion && $this->importacion->exists) {
                $this->importacion->update(['estado' => 'rechazado']);
            }

            throw $e;
        }
    }

    public function failed(\Throwable $exception): void
    {
        // Verificar que la importación existe antes de actualizar
        if ($this->importacion && $this->importacion->exists) {
            $this->importacion->update(['estado' => 'rechazado']);
        }

        Log::error("Job falló", [
            'importacion_id' => $this->importacion->id ?? 'desconocido',
            'error' => $exception->getMessage()
        ]);
    }
}
