<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use App\Models\UserSession;
use App\Models\User;
use Carbon\Carbon;
use Symfony\Component\HttpFoundation\Response;

class SessionControl
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        if (Auth::check()) {
            /** @var User $user */
            $user = Auth::user();
            $currentSessionId = Session::getId();

            // Limpiar sesiones inactivas
            UserSession::limpiarSesionesInactivas();

            // Actualizar sesión actual (excepto para verificaciones de estado)
            if (!$request->is('sessions/status') && !$request->is('session-limit')) {
                $wasNewSession = $this->updateCurrentSession($user, $currentSessionId, $request);

                // Si es una nueva sesión y excede el límite, redirigir después de asegurar registro
                if ($wasNewSession && $this->shouldRedirectToSessionLimit($user, $request)) {
                    // Pequeño delay para asegurar que la sesión se registre correctamente
                    // antes de aplicar límites
                    if ($this->isPageRequest($request)) {
                        return redirect()->route('auth.session-limit');
                    }
                }
            }
        }

        return $next($request);
    }

    /**
     * Actualiza la sesión actual del usuario
     */
    private function updateCurrentSession($user, $sessionId, $request)
    {
        $userAgent = $request->userAgent();
        $ipAddress = $request->ip();

        // Verificar si es una sesión nueva
        $existingSession = UserSession::where('user_id', $user->id)
            ->where('session_id', $sessionId)
            ->first();

        $wasNewSession = !$existingSession;

        UserSession::updateOrCreate(
            [
                'user_id' => $user->id,
                'session_id' => $sessionId
            ],
            [
                'ip_address' => $ipAddress,
                'user_agent' => $userAgent,
                'device_name' => $this->getDeviceName($userAgent),
                'location' => $this->getLocation($ipAddress),
                'last_activity' => Carbon::now(),
                'is_current' => true
            ]
        );

        // Desmarcar otras sesiones como actuales
        UserSession::where('user_id', $user->id)
            ->where('session_id', '!=', $sessionId)
            ->update(['is_current' => false]);

        return $wasNewSession;
    }

    /**
     * Determina si debe redirigir a la página de límite de sesiones
     */
    private function shouldRedirectToSessionLimit($user, $request)
    {
        // No redirigir para peticiones AJAX o API
        if ($request->expectsJson() || $request->ajax()) {
            return false;
        }

        // No redirigir si ya está en rutas relacionadas con sesiones
        if ($request->is('sessions/*') || $request->is('login') || $request->is('logout')) {
            return false;
        }

        // Verificar si excede el límite
        $sesionesActivas = $user->getSesionesActivasCount();
        $limiteSesiones = $user->getLimiteSesiones();

        return $sesionesActivas > $limiteSesiones;
    }

    /**
     * Determina si es una petición de página (no AJAX/API)
     */
    private function isPageRequest($request)
    {
        return !$request->expectsJson() &&
            !$request->ajax() &&
            $request->method() === 'GET';
    }

    /**
     * Extrae el nombre del dispositivo del User Agent
     */
    private function getDeviceName($userAgent)
    {
        if (preg_match('/Mobile|Android|iPhone|iPad/', $userAgent)) {
            if (strpos($userAgent, 'iPhone') !== false) return 'iPhone';
            if (strpos($userAgent, 'iPad') !== false) return 'iPad';
            if (strpos($userAgent, 'Android') !== false) return 'Android';
            return 'Dispositivo móvil';
        }

        if (strpos($userAgent, 'Chrome') !== false) return 'Chrome (Escritorio)';
        if (strpos($userAgent, 'Firefox') !== false) return 'Firefox (Escritorio)';
        if (strpos($userAgent, 'Safari') !== false) return 'Safari (Escritorio)';
        if (strpos($userAgent, 'Edge') !== false) return 'Edge (Escritorio)';

        return 'Navegador desconocido';
    }

    /**
     * Obtiene la ubicación aproximada basada en IP
     */
    private function getLocation($ipAddress)
    {
        if ($ipAddress === '127.0.0.1' || $ipAddress === '::1') {
            return 'Local';
        }

        return 'Ubicación externa';
    }
}
