<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Vodafone;
use App\Models\LogImportacionVodafone;
use App\Models\TipoDocumento;
use App\Jobs\VodafoneImportJob;
use Illuminate\Support\Facades\Log;

class VodafoneImportController extends Controller
{
    /**
     * Muestra vista previa del archivo Excel antes de importar
     */
    public function preview(Request $request)
    {
        $this->validateExcelFile($request);

        try {
            $rows = $this->processExcelFile($request->file('archivo'));
            $previewData = $this->generatePreviewData($rows);

            return redirect()->back()->with([
                'importData' => $previewData,
                'success' => 'Archivo procesado correctamente'
            ] + $previewData);
        } catch (\Throwable $e) {
            return back()->withErrors([
                'message' => 'Error leyendo archivo: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * Ejecuta la importación confirmada en background
     */
    public function importarConfirmado(Request $request)
    {
        Log::info('Iniciando importación confirmada', [
            'modo' => $request->modo,
            'tiene_archivo' => $request->hasFile('archivo'),
        ]);

        $this->validateImportRequest($request);

        try {
            $rows = $this->processExcelFile($request->file('archivo'));
            $datos = $this->prepareDataForImport($rows);

            $log = $this->createImportLog($request, count($datos));

            Log::info('📊 DATOS PARA IMPORTACIÓN', [
                'cantidad_total' => count($datos),
                'archivo' => $request->file('archivo')->getClientOriginalName(),
                'modo' => $request->modo,
                'primer_registro' => $datos[0] ?? null,
                'estructura_muestra' => array_keys($datos[0] ?? []),
            ]);

            // Lanzar el job en background
            dispatch(new VodafoneImportJob($datos, $request->modo, $log->id, $request->user()->id));

            return redirect()->back()->with([
                'success' => 'Importación en proceso',
                'log_id' => $log->id,
                'total_registros' => count($datos),
            ]);
        } catch (\Throwable $e) {
            Log::error('Error en importación confirmada: ' . $e->getMessage());
            return redirect()->back()->withErrors([
                'message' => 'Error procesando archivo: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * Obtiene el estado y errores de un log de importación
     */
    public function obtenerErroresLog($id)
    {
        $log = LogImportacionVodafone::findOrFail($id);

        return response()->json([
            'estado' => $log->estado,
            'errores' => json_decode($log->errores_json ?? '[]'),
        ]);
    }

    // ==========================================
    // MÉTODOS PRIVADOS REUTILIZABLES
    // ==========================================

    /**
     * Valida que el archivo sea Excel válido
     */
    private function validateExcelFile(Request $request)
    {
        $request->validate([
            'archivo' => 'required|file|mimes:xlsx,xls',
        ]);
    }

    /**
     * Valida la petición de importación confirmada
     */
    private function validateImportRequest(Request $request)
    {
        $request->validate([
            'archivo' => 'required|file|mimes:xlsx,xls',
            'modo' => 'nullable|string|in:omitir,actualizar',
        ]);
    }

    /**
     * Procesa un archivo Excel y retorna las filas estructuradas
     */
    private function processExcelFile($file)
    {
        $array = \Maatwebsite\Excel\Facades\Excel::toArray([], $file);
        $rows = $array[0] ?? [];

        if (empty($rows)) {
            return collect([]);
        }

        // Tomar los encabezados (primera fila)
        $headers = $rows[0];
        $headerMap = $this->createHeaderMap($headers);

        // Obtener listas de datos existentes para detectar duplicados
        $existingData = $this->getExistingData();

        return collect($rows)
            ->slice(1) // Omitir encabezados
            ->map(function ($row, $index) use ($headerMap, $existingData) {
                return $this->mapRowToStructure($row, $headerMap, $index, $existingData);
            })
            ->filter(function ($row) {
                return $this->isValidRow($row);
            });
    }

    /**
     * Obtiene datos existentes para comparación de duplicados
     */
    private function getExistingData()
    {
        return [
            'documentos' => Vodafone::pluck('numero_documento')->toArray(),
            'telefonos' => Vodafone::pluck('telefono_principal')->toArray(),
        ];
    }

    /**
     * Crea un mapa de encabezados del Excel a campos de la BD
     */
    private function createHeaderMap($headers)
    {
        // Mapear encabezados del Excel a nombres de campos de la BD
        $headerMapping = [
            'orden_trabajo_anterior' => ['orden_trabajo_anterior', 'orden trabajo anterior', 'orden anterior'],
            'origen_base' => ['origen_base', 'origen base', 'base origen'],
            'nombre_cliente' => ['nombre_cliente', 'nombre cliente', 'cliente', 'nombre'],
            'numero_documento' => ['numero_documento', 'documento', 'dni', 'cedula', 'numero documento'],
            'tipo_documento' => ['tipo_documento', 'tipo documento', 'tipo_doc', 'tipo', 'document_type'], // ✅ NUEVO
            'telefono_principal' => ['telefono_principal', 'telefono', 'telefono principal', 'tel principal'],
            'telefono_adicional' => ['telefono_adicional', 'telefono adicional', 'tel adicional', 'segundo telefono'],
            'correo_referencia' => ['correo_referencia', 'correo', 'email', 'correo referencia'],
            'direccion_historico' => ['direccion_historico', 'direccion', 'direccion historico'],
            'marca_base' => ['marca_base', 'marca', 'marca base'],
            'origen_motivo_cancelacion' => ['origen_motivo_cancelacion', 'motivo cancelacion', 'cancelacion'],
            'observaciones' => ['observaciones', 'observacion', 'notas', 'comentarios'],
        ];

        $map = [];

        foreach ($headers as $index => $header) {
            $headerLower = strtolower(trim($header));

            foreach ($headerMapping as $field => $possibleNames) {
                foreach ($possibleNames as $possibleName) {
                    if ($headerLower === strtolower($possibleName)) {
                        $map[$field] = $index;
                        break 2; // Salir de ambos bucles
                    }
                }
            }
        }

        return $map;
    }

    /**
     * Mapea una fila del Excel a la estructura de datos de la base de datos
     */
    private function mapRowToStructure($row, $headerMap, $index, $existingData)
    {
        // Estructura base para la base de datos
        $baseStructure = $this->mapRowToBaseStructure($row, $headerMap);

        // Agregar campos auxiliares solo para preview
        return array_merge($baseStructure, [
            'index' => $index + 2, // Solo para preview
            'duplicado' => $this->isDuplicate($baseStructure['numero_documento'], $baseStructure['telefono_principal'], $existingData),
        ]);
    }

    /**
     * Mapea una fila del Excel solo a la estructura de base de datos (sin campos auxiliares)
     */
    private function mapRowToBaseStructure($row, $headerMap)
    {
        $result = [];

        // Mapear cada campo usando el mapa de encabezados
        $fields = [
            'orden_trabajo_anterior',
            'origen_base',
            'nombre_cliente',
            'numero_documento',
            'tipo_documento', // ✅ NUEVO
            'telefono_principal',
            'telefono_adicional',
            'correo_referencia',
            'direccion_historico',
            'marca_base',
            'origen_motivo_cancelacion',
            'observaciones'
        ];

        foreach ($fields as $field) {
            $columnIndex = $headerMap[$field] ?? null;
            $result[$field] = ($columnIndex !== null && isset($row[$columnIndex]))
                ? trim($row[$columnIndex])
                : '';
        }

        // ✅ DETECCIÓN INTELIGENTE DE TIPO DE DOCUMENTO
        $result = $this->detectarTipoDocumento($result);

        return $result;
    }

    /**
     * Detecta automáticamente el tipo de documento basándose en el número
     */
    private function detectarTipoDocumento($row)
    {
        $numeroDocumento = trim($row['numero_documento'] ?? '');
        $tipoEspecificado = trim($row['tipo_documento'] ?? '');

        // Si ya se especificó el tipo, buscar su ID
        if ($tipoEspecificado) {
            $tipoId = $this->buscarTipoDocumentoPorCodigo($tipoEspecificado);
            if ($tipoId) {
                $row['tipo_documento_id'] = $tipoId;
                return $row;
            }
        }

        // Si no se especificó o no se encontró, detectar automáticamente
        if ($numeroDocumento) {
            $tipoDetectado = $this->detectarTipoPorPatron($numeroDocumento);
            if ($tipoDetectado) {
                $row['tipo_documento_id'] = $tipoDetectado;
                Log::info("Tipo de documento detectado automáticamente", [
                    'documento' => $numeroDocumento,
                    'tipo_detectado_id' => $tipoDetectado
                ]);
            }
        }

        return $row;
    }

    /**
     * Busca el ID del tipo de documento por código
     */
    private function buscarTipoDocumentoPorCodigo($codigo)
    {
        $tiposDocumento = $this->getTiposDocumento();

        foreach ($tiposDocumento as $tipo) {
            if (
                strtoupper($codigo) === strtoupper($tipo['codigo']) ||
                strtoupper($codigo) === strtoupper($tipo['nombre'])
            ) {
                return $tipo['id'];
            }
        }

        return null;
    }

    /**
     * Detecta el tipo de documento basándose en patrones
     */
    private function detectarTipoPorPatron($numeroDocumento)
    {
        $numero = preg_replace('/[^a-zA-Z0-9]/', '', $numeroDocumento); // Limpiar espacios y caracteres especiales
        $tiposDocumento = $this->getTiposDocumento();

        // Patrones de detección
        $patrones = [
            'DNI' => '/^[0-9]{8}$/', // 8 dígitos exactos
            'RUC' => '/^[0-9]{11}$/', // 11 dígitos exactos
            'CE' => '/^[0-9]{9,12}$/', // Entre 9 y 12 dígitos
            'NIE' => '/^[XYZ][0-9]{7}[A-Z]$/i', // X/Y/Z + 7 dígitos + letra
            'NIF' => '/^[0-9]{8}[A-Z]$/i', // 8 dígitos + letra
            'PASAPORTE' => '/^[A-Z0-9]{6,9}$/i', // 6-9 caracteres alfanuméricos
        ];

        foreach ($patrones as $codigoTipo => $patron) {
            if (preg_match($patron, $numero)) {
                // Buscar el ID de este tipo en la BD
                foreach ($tiposDocumento as $tipo) {
                    if ($tipo['codigo'] === $codigoTipo) {
                        Log::info("Patrón detectado", [
                            'documento' => $numeroDocumento,
                            'patron' => $patron,
                            'tipo' => $codigoTipo
                        ]);
                        return $tipo['id'];
                    }
                }
            }
        }

        // Si no se detecta ningún patrón, usar "OTROS"
        foreach ($tiposDocumento as $tipo) {
            if ($tipo['codigo'] === 'OTROS') {
                Log::info("Tipo de documento no detectado, usando OTROS", [
                    'documento' => $numeroDocumento
                ]);
                return $tipo['id'];
            }
        }

        return null;
    }

    /**
     * Obtiene los tipos de documento desde la BD con cache
     */
    private function getTiposDocumento()
    {
        static $tiposDocumento = null;

        if ($tiposDocumento === null) {
            $tiposDocumento = TipoDocumento::where('activo', true)
                ->orderBy('orden')
                ->get(['id', 'codigo', 'nombre'])
                ->toArray();
        }

        return $tiposDocumento;
    }

    /**
     * Determina si un registro es duplicado
     */
    private function isDuplicate($documento, $telefono, $existingData)
    {
        return ($documento && in_array($documento, $existingData['documentos']))
            || ($telefono && in_array($telefono, $existingData['telefonos']));
    }

    /**
     * Valida si una fila tiene datos mínimos requeridos
     */
    private function isValidRow($row)
    {
        return $row['numero_documento'] || $row['telefono_principal'] || $row['nombre_cliente'];
    }

    /**
     * Genera datos de vista previa con límites y estadísticas
     */
    private function generatePreviewData($rows)
    {
        $maxPreview = 1000;
        $truncado = $rows->count() > $maxPreview;
        $previewRows = $rows->take($maxPreview)->values()->toArray();

        $totalRegistros = $rows->count();
        $totalDuplicados = $rows->where('duplicado', true)->count();
        $totalNuevos = $totalRegistros - $totalDuplicados;

        return [
            'preview' => $previewRows,
            'truncado' => $truncado,
            'total' => $totalRegistros,
            'total_registros' => $totalRegistros,
            'total_duplicados' => $totalDuplicados,
            'total_nuevos' => $totalNuevos,
        ];
    }

    /**
     * Prepara los datos para importación (usando estructura base sin campos auxiliares)
     */
    private function prepareDataForImport($rows)
    {
        // Procesar archivo nuevamente pero solo con estructura base
        $array = \Maatwebsite\Excel\Facades\Excel::toArray([], request()->file('archivo'));
        $allRows = $array[0] ?? [];

        if (empty($allRows)) {
            return [];
        }

        // Tomar los encabezados (primera fila)
        $headers = $allRows[0];
        $headerMap = $this->createHeaderMap($headers);

        return collect($allRows)
            ->slice(1) // Omitir encabezados
            ->map(function ($row) use ($headerMap) {
                return $this->mapRowToBaseStructure($row, $headerMap); // Solo estructura de BD
            })
            ->filter(function ($row) {
                return $this->isValidRowForImport($row);
            })
            ->values()
            ->toArray();
    }

    /**
     * Valida si una fila tiene datos mínimos para importación
     */
    private function isValidRowForImport($row)
    {
        return !empty($row['numero_documento']) || !empty($row['telefono_principal']) || !empty($row['nombre_cliente']);
    }

    /**
     * Crea un registro de log para seguimiento de la importación
     */
    private function createImportLog(Request $request, $totalRecords)
    {
        return LogImportacionVodafone::create([
            'user_id' => $request->user()->id,
            'nombre_archivo' => $request->file('archivo')->getClientOriginalName(),
            'cantidad_registros' => $totalRecords,
        ]);
    }
}
