<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\StoreVodafoneRequest;
use App\Http\Requests\UpdateVodafoneRequest;
use App\Models\Vodafone;
use App\Models\User;
use App\Models\TipoDocumento;
use Illuminate\Http\Request;
use Inertia\Inertia;

class VodafoneCrudController extends Controller
{
    use AlertResponseTrait;
    // =======================
    // MÉTODOS PRINCIPALES DE VISTA
    // =======================

    /**
     * Display the vodafone index with all registros.
     */
    public function index(Request $request)
    {
        $vodafones = $this->getVodafones($request);
        $usuarios = $this->getUsuarios();
        $tiposDocumento = $this->getTiposDocumento();

        // Configuración de filtros para el frontend (similar a VodafoneController)
        $configuracion = [
            'filtros' => [],
            'busqueda' => [
                'campo' => $request->input('search_campo', ''),
                'valor' => $request->input('search_valor', ''),
            ],
            'paginacion' => [
                'por_pagina' => (int)$request->input('per_page', 20),
                'pagina_actual' => (int)$request->input('page', 1),
            ],
        ];

        // Si hay búsqueda, agregarla a la configuración
        if ($request->filled('search')) {
            $configuracion['busqueda']['valor'] = $request->input('search');
            $configuracion['busqueda']['campo'] = 'general'; // búsqueda general
        }

        return Inertia::render('GestionarVodafoneCrud', [
            'vodafones' => $vodafones,
            'usuarios' => $usuarios,
            'tiposDocumento' => $tiposDocumento,
            'configuracion' => $configuracion,
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    /**
     * Store a newly created vodafone in storage.
     */
    public function store(StoreVodafoneRequest $request)
    {
        try {
            $vodafone = $this->createVodafone($request->validated());

            return $this->redirectWithSuccess('Registro Vodafone creado correctamente', 'vodafone.crud.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al crear registro', $e);
        }
    }

    /**
     * Update the specified vodafone in storage.
     */
    public function update(UpdateVodafoneRequest $request, Vodafone $vodafone)
    {
        try {
            $this->updateVodafone($vodafone, $request->validated());

            return $this->redirectWithSuccess('Registro Vodafone actualizado correctamente', 'vodafone.crud.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al actualizar registro', $e);
        }
    }

    /**
     * Remove the specified vodafone from storage.
     */
    public function destroy(Vodafone $vodafone)
    {
        try {
            $vodafone->delete();
            return $this->redirectWithSuccess('Registro Vodafone eliminado correctamente', 'vodafone.crud.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al eliminar registro', $e);
        }
    }

    // =======================
    // MÉTODOS AUXILIARES PARA DATOS
    // =======================

    private function getVodafones($request = null)
    {
        $query = Vodafone::with(['user', 'asignado_a', 'tipoDocumento']);

        // Búsqueda por múltiples campos
        if ($request && $request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('nombre_cliente', 'LIKE', "%{$search}%")
                    ->orWhere('numero_documento', 'LIKE', "%{$search}%")
                    ->orWhere('telefono_principal', 'LIKE', "%{$search}%")
                    ->orWhere('telefono_adicional', 'LIKE', "%{$search}%")
                    ->orWhere('correo_referencia', 'LIKE', "%{$search}%");
            });
        }

        // Búsqueda específica por campo y valor (para compatibilidad futura)
        if ($request && $request->filled('search_campo') && $request->filled('search_valor')) {
            $campo = $request->search_campo;
            $valor = $request->search_valor;

            if (in_array($campo, ['nombre_cliente', 'numero_documento', 'telefono_principal', 'telefono_adicional', 'correo_referencia'])) {
                $query->where($campo, 'LIKE', "%{$valor}%");
            }
        }

        // Paginación configurable (por defecto 20, máximo 100)
        $perPage = $request && $request->filled('per_page')
            ? min((int)$request->per_page, 100)
            : 20;

        return $query->orderBy('created_at', 'desc')
            ->paginate($perPage)
            ->withQueryString(); // Mantiene los parámetros de búsqueda en los enlaces de paginación
    }

    private function getUsuarios()
    {
        return User::select('id', 'name')->get();
    }

    private function getTiposDocumento()
    {
        return TipoDocumento::select('id', 'nombre', 'codigo')->get();
    }

    // =======================
    // MÉTODOS AUXILIARES PARA CRUD
    // =======================

    private function createVodafone($data)
    {
        // Si trazabilidad viene como null o vacío, lo eliminamos para que use el valor por defecto de la BD
        if (!isset($data['trazabilidad']) || empty($data['trazabilidad'])) {
            unset($data['trazabilidad']);
        }

        return Vodafone::create($data);
    }

    private function updateVodafone($vodafone, $data)
    {
        $vodafone->update($data);
    }
}
