<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Models\User;
use App\Models\Reporte;
use App\Models\Cartera;
use App\Models\Role;
use App\Models\Plan;
use App\Models\TipoDocumento;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{
    use AlertResponseTrait;
    // =======================
    // MÉTODOS PRINCIPALES
    // =======================

    public function index()
    {
        // Obtener usuarios con relaciones y datos efectivos
        $users = $this->getUsersWithEffectiveData();

        // Obtener datos auxiliares para el formulario
        $formData = $this->getFormData();

        return Inertia::render('GestionarUsuarios', [
            'users' => $users,
            'carteras' => $formData['carteras'],
            'reportes' => $formData['reportes'],
            'roles' => $formData['roles'],
            'tiposDocumento' => $formData['tiposDocumento'],
            'planes' => $formData['planes'],
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    public function store(StoreUserRequest $request)
    {
        try {
            // Crear usuario básico
            $user = $this->createUser($request->validated());

            // Asignar roles
            $user->syncRoles($request->validated()['roles'] ?? []);

            // Aplicar lógica de herencia y asignaciones personalizadas
            $this->applyInheritanceLogic($user, $request->validated());

            return $this->redirectWithSuccess("Usuario creado correctamente.", 'users.index');
        } catch (\Exception $e) {
            Log::error('Error al crear usuario', [
                'message' => $e->getMessage(),
                'data' => $request->all()
            ]);
            return $this->redirectWithError('Ocurrió un error al crear el usuario. Intente nuevamente.');
        }
    }

    public function update(UpdateUserRequest $request, User $user)
    {
        try {
            // Actualizar datos básicos del usuario
            $this->updateUserBasicData($user, $request->validated());

            // Sincronizar roles
            $user->syncRoles($request->validated()['roles'] ?? []);

            // Aplicar lógica de herencia y asignaciones personalizadas
            $this->applyInheritanceLogic($user, $request->validated());

            return $this->redirectWithSuccess("Usuario actualizado correctamente.", 'users.index');
        } catch (\Exception $e) {
            Log::error('Error al actualizar usuario', [
                'user_id' => $user->id,
                'message' => $e->getMessage(),
                'data' => $request->all()
            ]);
            return $this->redirectWithError('Ocurrió un error al actualizar el usuario. Intente nuevamente.');
        }
    }

    public function destroy(Request $request, User $user)
    {
        try {
            $userName = $user->name;
            $user->delete();

            return $this->redirectWithSuccess("Usuario eliminado correctamente.", 'users.index');
        } catch (\Exception $e) {
            Log::error('Error al eliminar usuario', [
                'user_id' => $user->id,
                'message' => $e->getMessage()
            ]);
            return $this->redirectWithError('Ocurrió un error al eliminar el usuario. Intente nuevamente.');
        }
    }

    // =======================
    // MÉTODOS AUXILIARES PARA DATOS
    // =======================

    private function getUsersWithEffectiveData()
    {
        return User::with([
            'carteras',
            'reportes.cartera',
            'roles',
            'roles.carteras',
            'roles.reportes.cartera',
            'tipoDocumento',
        ])->get()->map(function ($user) {
            return [
                ...$user->toArray(),
                'effective_carteras' => $user->getEffectiveCarteras(),
                'effective_reportes' => $user->getEffectiveReportes(),
            ];
        });
    }

    private function getFormData()
    {
        return [
            'carteras' => Cartera::with('reportes')->get(),
            'reportes' => Reporte::all(),
            'roles' => Role::with(['carteras', 'reportes.cartera'])->get(),
            'tiposDocumento' => TipoDocumento::activos()->ordenado()->get(),
            'planes' => Plan::where('activo', true)->orderBy('orden')->get(),
        ];
    }

    // =======================
    // MÉTODOS AUXILIARES PARA CRUD
    // =======================

    private function createUser($data)
    {
        return User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => bcrypt($data['password']),
            'active' => $data['active'],
            'plan_id' => $data['plan_id'] ?? null,
            'tipo_documento_id' => $data['tipo_documento_id'] ?? null,
            'numero_documento' => $data['numero_documento'] ?? null,
            'department' => $data['department'] ?? null,
            'position' => $data['position'] ?? null,
            'phone' => $data['phone'] ?? null,
            'bio' => $data['bio'] ?? null,
        ]);
    }

    private function updateUserBasicData($user, $data)
    {
        $updateData = [
            'name' => $data['name'],
            'email' => $data['email'],
            'active' => $data['active'],
            'plan_id' => $data['plan_id'] ?? null,
            'tipo_documento_id' => $data['tipo_documento_id'] ?? null,
            'numero_documento' => $data['numero_documento'] ?? null,
            'department' => $data['department'] ?? null,
            'position' => $data['position'] ?? null,
            'phone' => $data['phone'] ?? null,
            'bio' => $data['bio'] ?? null,
        ];

        // Actualizar contraseña solo si se proporciona
        if (array_key_exists('password', $data) && $data['password']) {
            $updateData['password'] = bcrypt($data['password']);
        }

        $user->update($updateData);
    }

    // =======================
    // MÉTODOS AUXILIARES PARA HERENCIA DE PERMISOS
    // =======================

    private function applyInheritanceLogic($user, $data)
    {
        // Obtener IDs de carteras y reportes heredados de roles
        $inheritedData = $this->getInheritedDataFromRoles($user);

        // Filtrar asignaciones personalizadas (que no vienen del rol)
        $personalizedData = $this->getPersonalizedAssignments($data, $inheritedData);

        // Sincronizar solo las asignaciones personalizadas
        $user->carteras()->sync($personalizedData['carteras']);
        $user->reportes()->sync($personalizedData['reportes']);
    }

    private function getInheritedDataFromRoles($user)
    {
        return [
            'carteras' => $user->roles->flatMap->carteras->pluck('id')->unique(),
            'reportes' => $user->roles->flatMap->reportes->pluck('id')->unique(),
        ];
    }

    private function getPersonalizedAssignments($data, $inheritedData)
    {
        $carterasPersonalizadas = collect($data['carteras'] ?? [])
            ->filter(fn($id) => !$inheritedData['carteras']->contains($id));

        $reportesPersonalizados = collect($data['reportes'] ?? [])
            ->filter(fn($id) => !$inheritedData['reportes']->contains($id));

        return [
            'carteras' => $carterasPersonalizadas,
            'reportes' => $reportesPersonalizados,
        ];
    }
}
