<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use App\Models\UserSession;
use App\Models\User;
use Inertia\Inertia;

class SessionController extends Controller
{
    /**
     * Muestra las sesiones activas del usuario
     */
    public function index()
    {
        /** @var User $user */
        $user = Auth::user();
        $currentSessionId = Session::getId();

        // Limpiar sesiones inactivas
        UserSession::limpiarSesionesInactivas();

        $sesiones = $user->userSessions()
            ->orderBy('last_activity', 'desc')
            ->get()
            ->map(function ($session) use ($currentSessionId) {
                return [
                    'id' => $session->id,
                    'device_name' => $session->device_name,
                    'ip_address' => $session->ip_address,
                    'location' => $session->location,
                    'last_activity' => $session->last_activity,
                    'last_activity_human' => $session->last_activity->diffForHumans(),
                    'is_current' => $session->session_id === $currentSessionId,
                    'is_active' => $session->isActive(),
                ];
            });

        return Inertia::render('Profile/Sessions', [
            'sessions' => $sesiones,
            'user_plan' => $user->plan,
            'sessions_limit' => $user->getLimiteSesiones(),
            'active_sessions_count' => $user->getSesionesActivasCount(),
        ]);
    }

    /**
     * Verifica el estado de sesiones
     */
    public function getSessionStatus()
    {
        /** @var User $user */
        $user = Auth::user();

        if (!$user) {
            return response()->json(['error' => 'No autenticado'], 401);
        }

        // Limpiar sesiones inactivas
        UserSession::limpiarSesionesInactivas();

        $currentSessionId = Session::getId();
        $sesionesActivas = $user->getSesionesActivasCount();
        $limiteSesiones = $user->getLimiteSesiones();
        $excedeLimite = $sesionesActivas > $limiteSesiones;

        $shouldCloseThisSession = false;
        $isNewestSession = false;

        if ($excedeLimite) {
            // Obtener sesiones ordenadas por creación (más antiguas primero)
            $todasLasSesiones = $user->userSessions()
                ->activas()
                ->orderBy('created_at', 'asc')
                ->get();

            // Las sesiones a cerrar son las más antiguas
            $sesionesACerrar = $todasLasSesiones->take($sesionesActivas - $limiteSesiones);

            // Esta sesión debe cerrarse si está en las que deben cerrar
            $shouldCloseThisSession = $sesionesACerrar->contains('session_id', $currentSessionId);

            // Esta sesión es la más nueva si es la última en la lista
            $sesionMasNueva = $todasLasSesiones->last();
            $isNewestSession = $sesionMasNueva && $sesionMasNueva->session_id === $currentSessionId;
        }

        return response()->json([
            'active_sessions' => $sesionesActivas,
            'sessions_limit' => $limiteSesiones,
            'limit_exceeded' => $excedeLimite,
            'should_close_this_session' => $shouldCloseThisSession,
            'is_newest_session' => $isNewestSession
        ]);
    }
    /**
     * Cierra otras sesiones y mantiene la actual
     */
    public function closeOtherSessions(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        $currentSessionId = Session::getId();

        $cerradas = $user->cerrarOtrasSesiones($currentSessionId);

        return response()->json([
            'success' => true,
            'message' => "Se cerraron {$cerradas} sesiones.",
            'sessions_closed' => $cerradas
        ]);
    }

    /**
     * Confirma una acción en el límite de sesiones
     */
    public function confirmLimit(Request $request)
    {
        $action = $request->input('action');

        switch ($action) {
            case 'close_others':
                /** @var User $user */
                $user = Auth::user();
                $currentSessionId = Session::getId();
                $cerradas = $user->cerrarOtrasSesiones($currentSessionId);

                return response()->json([
                    'success' => true,
                    'message' => "Se cerraron {$cerradas} sesiones.",
                    'sessions_closed' => $cerradas
                ]);

            case 'cancel':
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();

                return response()->json([
                    'success' => true,
                    'redirect' => route('login')
                ]);

            default:
                return response()->json([
                    'success' => false,
                    'message' => 'Acción no válida'
                ], 400);
        }
    }

    /**
     * Cierra la sesión actual
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return response()->json([
            'success' => true,
            'redirect' => route('login')
        ]);
    }
}
