<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\StoreReporteRequest;
use App\Http\Requests\UpdateReporteRequest;
use App\Models\Reporte;
use App\Models\Cartera;
use Inertia\Inertia;

class ReportesController extends Controller
{
    use AlertResponseTrait;
    // =======================
    // MÉTODOS PRINCIPALES DE VISTA
    // =======================

    /**
     * Display the reports index with all reports and carteras.
     */
    public function index()
    {
        // Obtener reportes con relaciones
        $reportes = $this->getReportsWithRelations();

        // Obtener datos auxiliares para formularios
        $carteras = $this->getCarteras();

        return Inertia::render('GestionarReportes', [
            'reportes' => $reportes,
            'carteras' => $carteras,
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    /**
     * Store a newly created report in storage.
     */
    public function store(StoreReporteRequest $request)
    {
        try {
            $reporte = $this->createReport($request->validated());

            return $this->redirectWithSuccess('Reporte creado correctamente', 'reportes.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al crear reporte', $e);
        }
    }

    /**
     * Update the specified report in storage.
     */
    public function update(UpdateReporteRequest $request, Reporte $reporte)
    {
        try {
            $this->updateReport($reporte, $request->validated());

            return $this->redirectWithSuccess('Reporte actualizado correctamente', 'reportes.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al actualizar reporte', $e);
        }
    }

    /**
     * Remove the specified report from storage.
     */
    public function destroy(Reporte $reporte)
    {
        try {
            $reporte->delete();

            return $this->redirectWithSuccess('Reporte eliminado correctamente', 'reportes.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al eliminar reporte', $e);
        }
    }

    // =======================
    // MÉTODOS AUXILIARES PARA DATOS
    // =======================

    private function getReportsWithRelations()
    {
        return Reporte::with('cartera')->get();
    }

    private function getCarteras()
    {
        return Cartera::all();
    }

    // =======================
    // MÉTODOS AUXILIARES PARA CRUD
    // =======================

    private function createReport($data)
    {
        return Reporte::create($data);
    }

    private function updateReport($reporte, $data)
    {
        $reporte->update($data);
    }
}
