<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\StoreCarteraRequest;
use App\Http\Requests\UpdateCarteraRequest;
use App\Models\Cartera;
use App\Models\Reporte;
use Inertia\Inertia;

class CarterasController extends Controller
{
    use AlertResponseTrait;
    // =======================
    // MÉTODOS PRINCIPALES DE VISTA
    // =======================

    /**
     * Display the carteras index with all carteras and reportes.
     */
    public function index()
    {
        // Obtener carteras
        $carteras = $this->getCarteras();

        // Obtener reportes con relaciones
        $reportes = $this->getReportesWithRelations();

        return Inertia::render('GestionarCarteras', [
            'carteras' => $carteras,
            'reportes' => $reportes,
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    /**
     * Store a newly created cartera in storage.
     */
    public function store(StoreCarteraRequest $request)
    {
        try {
            $cartera = $this->createCartera($request->validated());

            return $this->redirectWithSuccess('Cartera creada correctamente', 'carteras.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al crear cartera', $e);
        }
    }

    /**
     * Update the specified cartera in storage.
     */
    public function update(UpdateCarteraRequest $request, Cartera $cartera)
    {
        try {
            $this->updateCartera($cartera, $request->validated());

            return $this->redirectWithSuccess('Cartera actualizada correctamente', 'carteras.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al actualizar cartera', $e);
        }
    }

    /**
     * Remove the specified cartera from storage.
     */
    public function destroy(Cartera $cartera)
    {
        try {
            // Verificar restricciones de negocio
            $this->validateDeletion($cartera);

            $cartera->delete();
            return $this->redirectWithSuccess('Cartera eliminada correctamente', 'carteras.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al eliminar cartera', $e);
        }
    }

    // =======================
    // MÉTODOS AUXILIARES PARA DATOS
    // =======================

    private function getCarteras()
    {
        return Cartera::all();
    }

    private function getReportesWithRelations()
    {
        return Reporte::with('cartera')->get();
    }

    // =======================
    // MÉTODOS AUXILIARES PARA CRUD
    // =======================

    private function createCartera($data)
    {
        return Cartera::create($data);
    }

    private function updateCartera($cartera, $data)
    {
        $cartera->update($data);
    }

    // =======================
    // VALIDACIONES DE NEGOCIO
    // =======================

    /**
     * Valida si una cartera puede ser eliminada
     */
    private function validateDeletion(Cartera $cartera)
    {
        if ($cartera->reportes()->count() > 0) {
            throw new \InvalidArgumentException('Esta Cartera tiene reportes asociados y no puede ser eliminada');
        }
    }
}
