<?php

namespace App\Exports;

use App\Models\EnriquecimientoImportacion;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class EnriquecimientoExport implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize
{
    protected $importacionId;

    public function __construct($importacionId)
    {
        $this->importacionId = $importacionId;
    }

    public function collection()
    {
        return EnriquecimientoImportacion::with(['numeros' => function ($query) {
            $query->orderBy('fecha_registro', 'asc');
        }])
            ->findOrFail($this->importacionId)
            ->numeros;
    }

    public function headings(): array
    {
        return [
            'ID',
            'Número Teléfono',
            'Número Normalizado',
            'Operador Origen',
            'Operador Actual',
            'Fecha Actualización Operador',
            'Estado',
            'Fecha Procesamiento',
            'Fecha Registro',
            'Detalles Error',
            'Datos Adicionales'
        ];
    }

    public function map($numero): array
    {
        return [
            $numero->id,
            $numero->numero_telefono,
            $numero->numero_normalizado,
            $numero->operador_origen,
            $numero->operador_actual,
            $numero->fecha_actualizacion_operador ? $numero->fecha_actualizacion_operador->format('Y-m-d H:i:s') : '',
            $this->getEstadoTexto($numero->estado),
            $numero->fecha_procesamiento ? $numero->fecha_procesamiento->format('Y-m-d H:i:s') : '',
            $numero->fecha_registro->format('Y-m-d H:i:s'),
            $numero->error_detalle,
            $numero->datos_adicionales ? json_encode($numero->datos_adicionales) : ''
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // Estilo para el encabezado
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['argb' => Color::COLOR_WHITE],
                    'size' => 12,
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['argb' => '4F46E5'], // Indigo
                ],
                'alignment' => [
                    'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                    'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
                ],
            ],
        ];
    }

    private function getEstadoTexto($estado)
    {
        return match ($estado) {
            'pendiente' => 'Pendiente',
            'procesando' => 'Procesando',
            'completado' => 'Completado',
            'error' => 'Error',
            'sin_datos' => 'Sin Datos',
            default => $estado
        };
    }
}
